/* 

	ADC System, by Mauro Grassi, September 2010.

*/

#include <math.h>
#include "HardwareProfile.h"
#include "Delay.h"
#include "main.h"
#include "adc.h"

#pragma udata adcdata

float 			adcSupplyVoltage;
float			adcDenominator;
float 			adcP;
unsigned char 	adcOpen;
unsigned char 	adcSave;

#pragma code usercode

float setADCSupplyVoltage(float x)
{
	if((x>=ADC_SUPPLY_MIN)&&(x<=ADC_SUPPLY_MAX))
	{
		adcSupplyVoltage=x;
	}
	return adcSupplyVoltage;
}

float getADCSupplyVoltage()
{
	return adcSupplyVoltage;
}

float readADC(unsigned int channel)
{
	float tmp;
	ADCON0=0x00;
	ADCON0|=((channel & 0x0F)<<2);
	ADCON0|=0x01;
	ADCON0bits.GO=1;
	while(ADCON0bits.GO);
	tmp=(float)(((unsigned int)ADRESH)<<8);
	tmp+=(float)((unsigned int)ADRESL);
	tmp*=(adcSupplyVoltage/adcDenominator);
	return tmp;
}

float readV(unsigned int channel)
{
	float f;
	
	f=readADC(channel);
	switch(channel)
	{
		default:
			break;
		case 0:
		case 1:
			/* low voltage */
			f*=(float)LOW_V_DIVIDERS_K;
			break;
			
		case 2:
		case 3:
			/* high voltage */
			f*=(float)HIGH_V_DIVIDERS_K;
			break;
	}	
	return f;
}

float getADCP(void)
{
	float f;	
	f=(ADC_FACTOR*readADC(ADCP_CH));
	if(f<ADCP_THRESHOLD)
	{

	}
	else
	{
		adcP=f;
	}
	return adcP;
}

float getADCReference(float denominator, float targetVoltage)
{
	float f;
	
	f=1.0;
	/* power on band gap reference */
	ANCON1bits.VBGEN=1;	
	/* For Best Accuracy, the band gap reference circuit needs to be enabled at least 10ms before the conversion is made */
	DelayMs(11);
	ADCON0=0x3D;
	ADCON0bits.GO=1;
	while(ADCON0bits.GO);
	f=(float)(((unsigned int)ADRESH)<<8);
	f+=(float)((unsigned int)ADRESL);
	f=(targetVoltage*denominator)/f;
	adcDenominator=denominator;
	/* power down the band gap reference */
	ANCON1bits.VBGEN=0;
	return f;
}

void setVGBIO(unsigned char x)
{
	if(x)
	{
		WDTCONbits.VBGOE=1;
	}
	else
	{
		WDTCONbits.VBGOE=0;		
	}
}

void calibrateADC(void)
{
	/* perform a calibration */
	ADCON1bits.ADCAL=1;
	ADCON0bits.GO=1;
	while(ADCON0bits.GO==1);	
	ADCON1bits.ADCAL=0;
}

unsigned char initADC(void)
{
	#if(USE_PMDIS)
		PMDIS0bits.ADCMD=0;
	#endif
	
	adcOpen=0;

	ADCON0=0x00;
	ADCON1=0xA6;
	/* Configure the Analog Channels */
	
	/* Configure ADCP pin = AN4 */
	ADCP_TRIS=1;
	ADCP_PCFG=0;
	
	calibrateADC();
	
	adcSupplyVoltage=getADCReference(ADC_DENOMINATOR, BAND_GAP_VOLTAGE);
	adcP=ADC_SUPPLY_DEFAULT;
	
	return 1;
}

unsigned char openADCObject(int physicalChannel)
{
	unsigned char result;
	
	result=0;
	
	if((physicalChannel>=0)&&(physicalChannel<=12))
	{	
		switch(physicalChannel)
		{	
			default:
				break;
		
			case 0:
				adcOpen|=ADC_CH0;
				TRISAbits.TRISA0=1;
				ANCON0bits.PCFG0=0;
				result=1;
				break;
		
			case 1:
				adcOpen|=ADC_CH1;
				TRISAbits.TRISA1=1;
				ANCON0bits.PCFG1=0;
				result=1;
				break;

			case 2:
				adcOpen|=ADC_CH2;		
				TRISAbits.TRISA2=1;
				ANCON0bits.PCFG2=0;
				result=1;
				break;

			case 3:
				adcOpen|=ADC_CH3;
				TRISAbits.TRISA3=1;
				ANCON0bits.PCFG3=0;
				result=1;
				break;

			case 4:
				break;
					
			case 5:
			case 6:
			case 7:
				/* these are unimplemented in 28 pin devices */
				break;
			
			case 8:
				//TRISBbits.TRISB2=1;
				//ANCON1bits.PCFG8=0;
				break;
			
			case 9:
				//TRISBbits.TRISB3=1;
				//ANCON1bits.PCFG9=0;
				break;
				
			case 10:
				//TRISBbits.TRISB1=1;
				//ANCON1bits.PCFG10=0;
				break;

			case 11:
				//TRISCbits.TRISC2=1;
				//ANCON1bits.PCFG11=0;
				break;
		
			case 12:
				//TRISBbits.TRISB0=1;
				//ANCON1bits.PCFG12=0;
				break;
		}
	}
	return result;	
}

void reopenCloseADCObject(unsigned char adcO, unsigned char on)
{
	if(adcO & ADC_CH0)
	{
		if(on)openADCObject(0); else closeADCObject(0);
	}
	
	if(adcO & ADC_CH1)
	{
		if(on)openADCObject(1); else closeADCObject(1);
	}

	if(adcO & ADC_CH2)
	{
		if(on)openADCObject(2); else closeADCObject(2);
	}

	if(adcO & ADC_CH3)
	{
		if(on)openADCObject(3); else closeADCObject(3);
	}
}

unsigned char closeADCObject(int physicalChannel)
{
	unsigned char result;
	
	result=0;
	if((physicalChannel>=0)&&(physicalChannel<=12))
	{	
		switch(physicalChannel)
		{	
			default:
				break;
		
			case 0:
				adcOpen&=~ADC_CH0;
				TRISAbits.TRISA0=1;
				ANCON0bits.PCFG0=1;
				result=1;
				break;
		
			case 1:
				adcOpen&=~ADC_CH1;
				TRISAbits.TRISA1=1;
				ANCON0bits.PCFG1=1;
				result=1;
				break;

			case 2:
				adcOpen&=~ADC_CH2;
				TRISAbits.TRISA2=1;
				ANCON0bits.PCFG2=1;
				result=1;
				break;

			case 3:
				adcOpen&=~ADC_CH3;
				TRISAbits.TRISA3=1;
				ANCON0bits.PCFG3=1;
				result=1;
				break;

			case 4:
				break;
		
			case 5:
			case 6:
			case 7:
				/* these are unimplemented in 28 pin devices */
				break;
			
			case 8:
				//TRISBbits.TRISB2=1;
				//ANCON1bits.PCFG8=1;
				break;
			
			case 9:
				//TRISBbits.TRISB3=1;
				//ANCON1bits.PCFG9=1;
				break;
				
			case 10:
				//TRISBbits.TRISB1=1;
				//ANCON1bits.PCFG10=1;
				break;

			case 11:
				//TRISCbits.TRISC2=1;
				//ANCON1bits.PCFG11=1;
				break;
		
			case 12:
				//TRISBbits.TRISB0=1;
				//ANCON1bits.PCFG12=1;
				break;
		}
	}	
	return result;
}

void powerADC(unsigned char on)
{
	if(on)
	{
	
	
	}
	else
	{
		#if(USE_PMDIS)
			PMDIS0bits.ADCMD=1;
		#endif
	}
}
